/*!---------------------------------------------------------
 * Copyright (C) Microsoft Corporation. All rights reserved.
 *--------------------------------------------------------*/
"use strict";
var try_all_1 = require("./try-all");
var path = require("path");
var advanced_1 = require("./advanced");
var events_1 = require("events");
var INTERNAL_ERROR = "internalError";
var ERROR_REPORTING_STARTED = "reportingStarted";
var ERROR_REPORTING_FINISHED = "reportingFinished";
var ErrorReporter = (function () {
    function ErrorReporter(config) {
        this._packageInfo = config.packageInfo;
        if (config.appRoot === "" || path.isAbsolute(config.appRoot)) {
            this._packageInfo.packageJsonFullPath = config.appRoot;
        }
        else {
            throw new Error(("Invalid app root \"" + config.appRoot + "\". ") +
                "You must provide an absolute path or an empty string");
        }
        this._exeName = config.exeName;
        this.emitter = new events_1.EventEmitter();
        this.channels = [];
    }
    ErrorReporter.prototype.addReportingChannel = function (channel) {
        this.channels.push(channel);
        channel.registerListeners(this);
    };
    ErrorReporter.prototype.removeReportingChannel = function (channel) {
        var index = this.channels.indexOf(channel);
        if (index !== -1) {
            this.channels.splice(index, 1);
        }
    };
    Object.defineProperty(ErrorReporter.prototype, "appName", {
        get: function () {
            return this._packageInfo.name;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(ErrorReporter.prototype, "appVersion", {
        get: function () {
            return this._packageInfo.version;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(ErrorReporter.prototype, "appRoot", {
        get: function () {
            return this._packageInfo.packageJsonFullPath;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(ErrorReporter.prototype, "branch", {
        get: function () {
            return this._packageInfo.branch;
        },
        enumerable: true,
        configurable: true
    });
    ErrorReporter.prototype.reportError = function (err, errorType) {
        var _this = this;
        this.emitReportingStarted();
        var stack;
        try {
            stack = new advanced_1.ErrorInformation({ rawError: err,
                packageInfo: this._packageInfo,
                errorType: errorType,
                exeName: this._exeName,
                errorReporter: this });
        }
        catch (e) {
            try {
                this.emitInternalError(e, err, "");
            }
            catch (e1) {
            }
            if (!stack) {
                return Promise.resolve({});
            }
        }
        var reportPromises = [];
        var rejectionFunctions = [];
        var timeoutLimits = [];
        this.channels.forEach(function (channel) {
            var prom;
            try {
                prom = channel.report(stack);
            }
            catch (e) {
                prom = Promise.reject(e);
            }
            var timeoutPromise = new Promise(function (resolve, reject) {
                rejectionFunctions.push(reject);
                timeoutLimits.push(channel.timeout);
            });
            var compositeProm = Promise.race([prom, timeoutPromise]);
            reportPromises.push(compositeProm);
        });
        for (var i = 0; i < rejectionFunctions.length; i++) {
            if (timeoutLimits[i] >= 0) {
                setTimeout(rejectionFunctions[i], timeoutLimits[i], "timed out after " + timeoutLimits[i] + " ms");
            }
        }
        return try_all_1.tryAll(reportPromises)
            .then(function (results) {
            _this.reportInternalFailures(results, _this.channels);
            _this.emitReportingFinished(results);
            return results;
        })
            .catch(function (reason) {
            return reason;
        });
    };
    ;
    ErrorReporter.prototype.reportInternalFailures = function (results, channels) {
        for (var i = 0; i < channels.length; i++) {
            if (channels[i].reportOnFailure && !results.resolved[i]) {
                this.emitInternalError(results.failReasons[i], null, null);
            }
        }
    };
    ErrorReporter.prototype.onInternalError = function (callback) {
        this.emitter.on(INTERNAL_ERROR, callback);
    };
    ErrorReporter.prototype.emitInternalError = function (internalError, errorBeingProcessed, problemString) {
        if (this.emitter.listenerCount(INTERNAL_ERROR) > 0) {
            var proc = new advanced_1.ErrorInformation({ rawError: internalError });
            var stack = void 0;
            try {
                stack = proc.getVerboseStack();
            }
            catch (e) {
                stack = "UNKNOWN\tUNKNOWN";
            }
            var info = {
                props: {
                    InternalErrorStack: stack,
                    InternalErrorType: proc.errorName
                },
                PIIProps: {
                    InternalErrorMessage: proc.errorMessage,
                    RawInternalError: internalError,
                    OriginalError: errorBeingProcessed,
                    ProblemString: problemString
                }
            };
            this.emitter.emit(INTERNAL_ERROR, info);
        }
    };
    ErrorReporter.prototype.onReportingStarted = function (callback) {
        this.emitter.on(ERROR_REPORTING_STARTED, callback);
    };
    ErrorReporter.prototype.emitReportingStarted = function () {
        return this.emitter.emit(ERROR_REPORTING_STARTED);
    };
    ErrorReporter.prototype.onReportingFinished = function (callback) {
        this.emitter.on(ERROR_REPORTING_FINISHED, callback);
    };
    ErrorReporter.prototype.emitReportingFinished = function (results) {
        return this.emitter.emit(ERROR_REPORTING_FINISHED, results);
    };
    return ErrorReporter;
}());
exports.ErrorReporter = ErrorReporter;

// SIG // Begin signature block
// SIG // MIIiAgYJKoZIhvcNAQcCoIIh8zCCIe8CAQExDzANBglg
// SIG // hkgBZQMEAgEFADB3BgorBgEEAYI3AgEEoGkwZzAyBgor
// SIG // BgEEAYI3AgEeMCQCAQEEEBDgyQbOONQRoqMAEEvTUJAC
// SIG // AQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg
// SIG // VqFjpJKaMYkNTH+A8hxvRcuZ4HxcYnZF2qvRczMcC3qg
// SIG // gguEMIIFDDCCA/SgAwIBAgITMwAAAU/nxmLJRvSpfwAA
// SIG // AAABTzANBgkqhkiG9w0BAQsFADB+MQswCQYDVQQGEwJV
// SIG // UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
// SIG // UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBv
// SIG // cmF0aW9uMSgwJgYDVQQDEx9NaWNyb3NvZnQgQ29kZSBT
// SIG // aWduaW5nIFBDQSAyMDEwMB4XDTE2MTExNzIxNTkxNFoX
// SIG // DTE4MDIxNzIxNTkxNFowgYMxCzAJBgNVBAYTAlVTMRMw
// SIG // EQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
// SIG // b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
// SIG // b24xDTALBgNVBAsTBE1PUFIxHjAcBgNVBAMTFU1pY3Jv
// SIG // c29mdCBDb3Jwb3JhdGlvbjCCASIwDQYJKoZIhvcNAQEB
// SIG // BQADggEPADCCAQoCggEBALSJkIp2DKzFPaG8gejXUAi7
// SIG // WmmnrsfMuFu9OA8hUUYII9G9Iplztyjv9werK/lI96UU
// SIG // BalNJBttYZacg50zwCx5mfBd8+zVRN+lzVn6dq0GP8ZK
// SIG // A+bqLZQQwmZv/WLQUIrRIpEjl2U0ocSaAl6S4rh8KHjC
// SIG // XuNG98rvOFeIrFYGWMqYG8eo1142rppFMOO2tz0Ywqcn
// SIG // sHlNU9gy5najUCr2P7q7QR4khk2Ym3qe9izivJBgZfuW
// SIG // afrgE2jiiSoudRidwm24IfSZXip/MaCnDczvTxFu/klC
// SIG // TmHeN8srnkDvzqtecAhzpXJ7t0bOWzJTXYo5UW+w16h4
// SIG // 3aMob17uQh8CAwEAAaOCAXswggF3MB8GA1UdJQQYMBYG
// SIG // CisGAQQBgjc9BgEGCCsGAQUFBwMDMB0GA1UdDgQWBBSS
// SIG // DqERcbYvVQ3DD/TJXHQuWgxqvzBSBgNVHREESzBJpEcw
// SIG // RTENMAsGA1UECxMETU9QUjE0MDIGA1UEBRMrMjMwODY1
// SIG // K2I0YjEyODc4LWUyOTMtNDNlOS1iMjFlLTdkMzA3MTlk
// SIG // NDUyZjAfBgNVHSMEGDAWgBTm/F97uyIAWORyTrX0IXQj
// SIG // MubvrDBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3Js
// SIG // Lm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9N
// SIG // aWNDb2RTaWdQQ0FfMjAxMC0wNy0wNi5jcmwwWgYIKwYB
// SIG // BQUHAQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3
// SIG // Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY0NvZFNp
// SIG // Z1BDQV8yMDEwLTA3LTA2LmNydDAMBgNVHRMBAf8EAjAA
// SIG // MA0GCSqGSIb3DQEBCwUAA4IBAQARwLlqQzX+hbE4gA24
// SIG // dHUsjXFJuDxB7S294gY2kHYrieW2gemG4xiMlpMyQqen
// SIG // 9rc0JTnQxnpD6tt3cMzU6ULrqMVVAnV4ntgQvgcbypRj
// SIG // +zwPce7UVWfXKfZqvwe85Y+C6vih4nMzPxSSXqG6ZVc3
// SIG // RYWAXNYsxos8aoejBxHL7NlxOCA1JD1UvcxVifR3EJvl
// SIG // maoPrsY8jp/aAE/PfeGzwCD3UjH3IkjVYY1/JgIZfYPM
// SIG // YZMq3bjc4gizp9/hBHdgyPSsNcsx5Wx1UcU3exNqTpVr
// SIG // j0CeyE0+rr1ZxSDWwnitt2RbO9DAFXA0FOXFnqYcx/T/
// SIG // 9stIweKN827OqrSgMIIGcDCCBFigAwIBAgIKYQxSTAAA
// SIG // AAAAAzANBgkqhkiG9w0BAQsFADCBiDELMAkGA1UEBhMC
// SIG // VVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcT
// SIG // B1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jw
// SIG // b3JhdGlvbjEyMDAGA1UEAxMpTWljcm9zb2Z0IFJvb3Qg
// SIG // Q2VydGlmaWNhdGUgQXV0aG9yaXR5IDIwMTAwHhcNMTAw
// SIG // NzA2MjA0MDE3WhcNMjUwNzA2MjA1MDE3WjB+MQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMSgwJgYDVQQDEx9NaWNyb3NvZnQg
// SIG // Q29kZSBTaWduaW5nIFBDQSAyMDEwMIIBIjANBgkqhkiG
// SIG // 9w0BAQEFAAOCAQ8AMIIBCgKCAQEA6Q5kUHlntcTj/QkA
// SIG // TJ6UrPdWaOpE2M/FWE+ppXZ8bUW60zmStKQe+fllguQX
// SIG // 0o/9RJwI6GWTzixVhL99COMuK6hBKxi3oktuSUxrFQfe
// SIG // 0dLCiR5xlM21f0u0rwjYzIjWaxeUOpPOJj/s5v40mFfV
// SIG // HV1J9rIqLtWFu1k/+JC0K4N0yiuzO0bj8EZJwRdmVMkc
// SIG // vR3EVWJXcvhnuSUgNN5dpqWVXqsogM3Vsp7lA7Vj07IU
// SIG // yMHIiiYKWX8H7P8O7YASNUwSpr5SW/Wm2uCLC0h31oVH
// SIG // 1RC5xuiq7otqLQVcYMa0KlucIxxfReMaFB5vN8sZM4Bq
// SIG // iU2jamZjeJPVMM+VHwIDAQABo4IB4zCCAd8wEAYJKwYB
// SIG // BAGCNxUBBAMCAQAwHQYDVR0OBBYEFOb8X3u7IgBY5HJO
// SIG // tfQhdCMy5u+sMBkGCSsGAQQBgjcUAgQMHgoAUwB1AGIA
// SIG // QwBBMAsGA1UdDwQEAwIBhjAPBgNVHRMBAf8EBTADAQH/
// SIG // MB8GA1UdIwQYMBaAFNX2VsuP6KJcYmjRPZSQW9fOmhjE
// SIG // MFYGA1UdHwRPME0wS6BJoEeGRWh0dHA6Ly9jcmwubWlj
// SIG // cm9zb2Z0LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1Jv
// SIG // b0NlckF1dF8yMDEwLTA2LTIzLmNybDBaBggrBgEFBQcB
// SIG // AQROMEwwSgYIKwYBBQUHMAKGPmh0dHA6Ly93d3cubWlj
// SIG // cm9zb2Z0LmNvbS9wa2kvY2VydHMvTWljUm9vQ2VyQXV0
// SIG // XzIwMTAtMDYtMjMuY3J0MIGdBgNVHSAEgZUwgZIwgY8G
// SIG // CSsGAQQBgjcuAzCBgTA9BggrBgEFBQcCARYxaHR0cDov
// SIG // L3d3dy5taWNyb3NvZnQuY29tL1BLSS9kb2NzL0NQUy9k
// SIG // ZWZhdWx0Lmh0bTBABggrBgEFBQcCAjA0HjIgHQBMAGUA
// SIG // ZwBhAGwAXwBQAG8AbABpAGMAeQBfAFMAdABhAHQAZQBt
// SIG // AGUAbgB0AC4gHTANBgkqhkiG9w0BAQsFAAOCAgEAGnTv
// SIG // V08pe8QWhXi4UNMi/AmdrIKX+DT/KiyXlRLl5L/Pv5PI
// SIG // 4zSp24G43B4AvtI1b6/lf3mVd+UC1PHr2M1OHhthosJa
// SIG // IxrwjKhiUUVnCOM/PB6T+DCFF8g5QKbXDrMhKeWloWmM
// SIG // IpPMdJjnoUdD8lOswA8waX/+0iUgbW9h098H1dlyACxp
// SIG // hnY9UdumOUjJN2FtB91TGcun1mHCv+KDqw/ga5uV1n0o
// SIG // UbCJSlGkmmzItx9KGg5pqdfcwX7RSXCqtq27ckdjF/qm
// SIG // 1qKmhuyoEESbY7ayaYkGx0aGehg/6MUdIdV7+QIjLcVB
// SIG // y78dTMgW77Gcf/wiS0mKbhXjpn92W9FTeZGFndXS2z1z
// SIG // NfM8rlSyUkdqwKoTldKOEdqZZ14yjPs3hdHcdYWch8Za
// SIG // V4XCv90Nj4ybLeu07s8n07VeafqkFgQBpyRnc89NT7be
// SIG // BVaXevfpUk30dwVPhcbYC/GO7UIJ0Q124yNWeCImNr7K
// SIG // sYxuqh3khdpHM2KPpMmRM19xHkCvmGXJIuhCISWKHC1g
// SIG // 2TeJQYkqFg/XYTyUaGBS79ZHmaCAQO4VgXc+nOBTGBpQ
// SIG // HTiVmx5mMxMnORd4hzbOTsNfsvU9R1O24OXbC2E9KteS
// SIG // LM43Wj5AQjGkHxAIwlacvyRdUQKdannSF9PawZSOB3sl
// SIG // cUSrBmrm1MbfI5qWdcUxghXWMIIV0gIBATCBlTB+MQsw
// SIG // CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQ
// SIG // MA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9z
// SIG // b2Z0IENvcnBvcmF0aW9uMSgwJgYDVQQDEx9NaWNyb3Nv
// SIG // ZnQgQ29kZSBTaWduaW5nIFBDQSAyMDEwAhMzAAABT+fG
// SIG // YslG9Kl/AAAAAAFPMA0GCWCGSAFlAwQCAQUAoIHEMBkG
// SIG // CSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisGAQQB
// SIG // gjcCAQsxDjAMBgorBgEEAYI3AgEVMC8GCSqGSIb3DQEJ
// SIG // BDEiBCBhR3t37tQ1SE6cg3+uuFJLl+ZFxhZ2crunZk0t
// SIG // QbdR1zBYBgorBgEEAYI3AgEMMUowSKAugCwAZQByAHIA
// SIG // bwByAC0AcgBlAHAAbwByAHQAZQByAC0AYgBhAHMAZQAu
// SIG // AGoAc6EWgBRodHRwOi8vbWljcm9zb2Z0LmNvbTANBgkq
// SIG // hkiG9w0BAQEFAASCAQBIY6Mbm6cJfw7E/tncX3q9DXuw
// SIG // C6hRbIdw+8jc3ozX7ih40VPmapv0oUOyuS06ENRHumkg
// SIG // SdwEPExNxPYia2x0+JX9iBszSK0bDNmwT9yZDKa0xohx
// SIG // tre/iv/j7GHjRPIU1ynzoBjFRm1ct7UQWr/mKV03olAz
// SIG // 8O/Ljx3STi21udhhvxMWMLPFvWb0KvyxN2Eapq08J9oZ
// SIG // lj0qx9XR8eEo1p/10bYQY4iObpdQN6Ow57efN2SFsfEr
// SIG // 2yUvcJMfafxm2GDBxpBu+DirQvsPAjtV6R95K4rKLiQK
// SIG // m/iWjOHjdb5ibYUq2YiXTm1/i4rvoBypeVs3csw8I/O3
// SIG // gWec+r8noYITSjCCE0YGCisGAQQBgjcDAwExghM2MIIT
// SIG // MgYJKoZIhvcNAQcCoIITIzCCEx8CAQMxDzANBglghkgB
// SIG // ZQMEAgEFADCCAT0GCyqGSIb3DQEJEAEEoIIBLASCASgw
// SIG // ggEkAgEBBgorBgEEAYRZCgMBMDEwDQYJYIZIAWUDBAIB
// SIG // BQAEIIup0Zs8BRssE4Ef6dK8epMmOV0gLVsrkpVOV1Mj
// SIG // s+7uAgZYPzFKOq4YEzIwMTYxMjI4MDA1ODAzLjU1OVow
// SIG // BwIBAYACAfSggbmkgbYwgbMxCzAJBgNVBAYTAlVTMRMw
// SIG // EQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
// SIG // b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
// SIG // b24xDTALBgNVBAsTBE1PUFIxJzAlBgNVBAsTHm5DaXBo
// SIG // ZXIgRFNFIEVTTjpGNTI4LTM3NzctOEE3NjElMCMGA1UE
// SIG // AxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2VydmljZaCC
// SIG // Ds0wggZxMIIEWaADAgECAgphCYEqAAAAAAACMA0GCSqG
// SIG // SIb3DQEBCwUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UE
// SIG // CBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEe
// SIG // MBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMTIw
// SIG // MAYDVQQDEylNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
// SIG // ZSBBdXRob3JpdHkgMjAxMDAeFw0xMDA3MDEyMTM2NTVa
// SIG // Fw0yNTA3MDEyMTQ2NTVaMHwxCzAJBgNVBAYTAlVTMRMw
// SIG // EQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
// SIG // b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
// SIG // b24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1w
// SIG // IFBDQSAyMDEwMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8A
// SIG // MIIBCgKCAQEAqR0NvHcRijog7PwTl/X6f2mUa3RUENWl
// SIG // CgCChfvtfGhLLF/Fw+Vhwna3PmYrW/AVUycEMR9BGxqV
// SIG // Hc4JE458YTBZsTBED/FgiIRUQwzXTbg4CLNC3ZOs1nMw
// SIG // VyaCo0UN0Or1R4HNvyRgMlhgRvJYR4YyhB50YWeRX4FU
// SIG // sc+TTJLBxKZd0WETbijGGvmGgLvfYfxGwScdJGcSchoh
// SIG // iq9LZIlQYrFd/XcfPfBXday9ikJNQFHRD5wGPmd/9WbA
// SIG // A5ZEfu/QS/1u5ZrKsajyeioKMfDaTgaRtogINeh4HLDp
// SIG // mc085y9Euqf03GS9pAHBIAmTeM38vMDJRF1eFpwBBU8i
// SIG // TQIDAQABo4IB5jCCAeIwEAYJKwYBBAGCNxUBBAMCAQAw
// SIG // HQYDVR0OBBYEFNVjOlyKMZDzQ3t8RhvFM2hahW1VMBkG
// SIG // CSsGAQQBgjcUAgQMHgoAUwB1AGIAQwBBMAsGA1UdDwQE
// SIG // AwIBhjAPBgNVHRMBAf8EBTADAQH/MB8GA1UdIwQYMBaA
// SIG // FNX2VsuP6KJcYmjRPZSQW9fOmhjEMFYGA1UdHwRPME0w
// SIG // S6BJoEeGRWh0dHA6Ly9jcmwubWljcm9zb2Z0LmNvbS9w
// SIG // a2kvY3JsL3Byb2R1Y3RzL01pY1Jvb0NlckF1dF8yMDEw
// SIG // LTA2LTIzLmNybDBaBggrBgEFBQcBAQROMEwwSgYIKwYB
// SIG // BQUHMAKGPmh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9w
// SIG // a2kvY2VydHMvTWljUm9vQ2VyQXV0XzIwMTAtMDYtMjMu
// SIG // Y3J0MIGgBgNVHSABAf8EgZUwgZIwgY8GCSsGAQQBgjcu
// SIG // AzCBgTA9BggrBgEFBQcCARYxaHR0cDovL3d3dy5taWNy
// SIG // b3NvZnQuY29tL1BLSS9kb2NzL0NQUy9kZWZhdWx0Lmh0
// SIG // bTBABggrBgEFBQcCAjA0HjIgHQBMAGUAZwBhAGwAXwBQ
// SIG // AG8AbABpAGMAeQBfAFMAdABhAHQAZQBtAGUAbgB0AC4g
// SIG // HTANBgkqhkiG9w0BAQsFAAOCAgEAB+aIUQ3ixuCYP4Fx
// SIG // Az2do6Ehb7Prpsz1Mb7PBeKp/vpXbRkws8LFZslq3/Xn
// SIG // 8Hi9x6ieJeP5vO1rVFcIK1GCRBL7uVOMzPRgEop2zEBA
// SIG // QZvcXBf/XPleFzWYJFZLdO9CEMivv3/Gf/I3fVo/HPKZ
// SIG // eUqRUgCvOA8X9S95gWXZqbVr5MfO9sp6AG9LMEQkIjzP
// SIG // 7QOllo9ZKby2/QThcJ8ySif9Va8v/rbljjO7Yl+a21dA
// SIG // 6fHOmWaQjP9qYn/dxUoLkSbiOewZSnFjnXshbcOco6I8
// SIG // +n99lmqQeKZt0uGc+R38ONiU9MalCpaGpL2eGq4EQoO4
// SIG // tYCbIjggtSXlZOz39L9+Y1klD3ouOVd2onGqBooPiRa6
// SIG // YacRy5rYDkeagMXQzafQ732D8OE7cQnfXXSYIghh2rBQ
// SIG // Hm+98eEA3+cxB6STOvdlR3jo+KhIq/fecn5ha293qYHL
// SIG // pwmsObvsxsvYgrRyzR30uIUBHoD7G4kqVDmyW9rIDVWZ
// SIG // eodzOwjmmC3qjeAzLhIp9cAvVCch98isTtoouLGp25ay
// SIG // p0Kiyc8ZQU3ghvkqmqMRZjDTu3QyS99je/WZii8bxyGv
// SIG // WbWu3EQ8l1Bx16HSxVXjad5XwdHeMMD9zOZN+w2/XU/p
// SIG // nR4ZOC+8z1gFLu8NoFA12u8JJxzVs341Hgi62jbb01+P
// SIG // 3nSISRIwggTaMIIDwqADAgECAhMzAAAAsIbpkiAKXer4
// SIG // AAAAAACwMA0GCSqGSIb3DQEBCwUAMHwxCzAJBgNVBAYT
// SIG // AlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQH
// SIG // EwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29y
// SIG // cG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1l
// SIG // LVN0YW1wIFBDQSAyMDEwMB4XDTE2MDkwNzE3NTY1NloX
// SIG // DTE4MDkwNzE3NTY1NlowgbMxCzAJBgNVBAYTAlVTMRMw
// SIG // EQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
// SIG // b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
// SIG // b24xDTALBgNVBAsTBE1PUFIxJzAlBgNVBAsTHm5DaXBo
// SIG // ZXIgRFNFIEVTTjpGNTI4LTM3NzctOEE3NjElMCMGA1UE
// SIG // AxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2VydmljZTCC
// SIG // ASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAPDl
// SIG // 8I2UWama12wJB3cUIVldtz14uDWt9TAx/87h/9bpGsmD
// SIG // mCP7kBDBkuw6vFBskL1c8WutTKL1YaWH4SmsozwS33/E
// SIG // AILOvbQKX8u525q3GS1a7QcArVT9M6z2+BIkSiRnV9q4
// SIG // S6GsD4D5Ob0gDbWnxCxL8aFW5vIvJ7kmwTMl2SsMMHJQ
// SIG // hGuSapWM8voITbguJNVD09zr83RqiLPJtGDcyWWtXwnl
// SIG // EWoUNMHkVwYSRuagvYRaJEJD9rz0LFWZWo2ziKaqEpQi
// SIG // voa1SomMqkewoMUfG4+c8GJtVJ0y06nEm1PpuXI7vxee
// SIG // r1hcncYrUCG72JbMNu+NpnPZrbtDcBUCAwEAAaOCARsw
// SIG // ggEXMB0GA1UdDgQWBBT8qAdS2e8p/phDPBVcdF63f3Pg
// SIG // rzAfBgNVHSMEGDAWgBTVYzpcijGQ80N7fEYbxTNoWoVt
// SIG // VTBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3JsLm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNU
// SIG // aW1TdGFQQ0FfMjAxMC0wNy0wMS5jcmwwWgYIKwYBBQUH
// SIG // AQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3Lm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1RpbVN0YVBD
// SIG // QV8yMDEwLTA3LTAxLmNydDAMBgNVHRMBAf8EAjAAMBMG
// SIG // A1UdJQQMMAoGCCsGAQUFBwMIMA0GCSqGSIb3DQEBCwUA
// SIG // A4IBAQCS/YD3db1lkD4XzdUVg8WOApE7rE59YO+POQaT
// SIG // kOMPqkU9iT8C/exvuFCQCWyE0jchmbHBhnYd5A3Gjv8Y
// SIG // 4KL6RrWFd3RuTgxERdC80rYEfnBLD9arB2lWSnZk6Pmq
// SIG // R6Ojhtz0z0S4SRBbZ4/R2nmuOt2HUwboj2r8f9fko4ZF
// SIG // FQ2RYiI6kErj2GgBScktDmt4bBEC9I9NEWBzqAMhDZfj
// SIG // D3J1u/3oDrjr/CBBnY6T/cUzcz45er1qNB45R2IRtRzv
// SIG // V/Rzg3GfqBRlvuQDjk1txAEzG35X7KHzHJepOKf0qDo5
// SIG // +Yk1/F2jIq4KvY0MqzESbum9YRbs9Bk6JIViX6myoYID
// SIG // djCCAl4CAQEwgeOhgbmkgbYwgbMxCzAJBgNVBAYTAlVT
// SIG // MRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdS
// SIG // ZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9y
// SIG // YXRpb24xDTALBgNVBAsTBE1PUFIxJzAlBgNVBAsTHm5D
// SIG // aXBoZXIgRFNFIEVTTjpGNTI4LTM3NzctOEE3NjElMCMG
// SIG // A1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vydmlj
// SIG // ZaIlCgEBMAkGBSsOAwIaBQADFQC8hPudWxLaxzaFO4hm
// SIG // nqMWGNXMHKCBwjCBv6SBvDCBuTELMAkGA1UEBhMCVVMx
// SIG // EzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1Jl
// SIG // ZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3Jh
// SIG // dGlvbjENMAsGA1UECxMETU9QUjEnMCUGA1UECxMebkNp
// SIG // cGhlciBOVFMgRVNOOjU3RjYtQzFFMC01NTRDMSswKQYD
// SIG // VQQDEyJNaWNyb3NvZnQgVGltZSBTb3VyY2UgTWFzdGVy
// SIG // IENsb2NrMA0GCSqGSIb3DQEBBQUAAgUA3A2ECDAiGA8y
// SIG // MDE2MTIyODAwMjEyOFoYDzIwMTYxMjI5MDAyMTI4WjB0
// SIG // MDoGCisGAQQBhFkKBAExLDAqMAoCBQDcDYQIAgEAMAcC
// SIG // AQACAgO/MAcCAQACAhfqMAoCBQDcDtWIAgEAMDYGCisG
// SIG // AQQBhFkKBAIxKDAmMAwGCisGAQQBhFkKAwGgCjAIAgEA
// SIG // AgMW42ChCjAIAgEAAgMHoSAwDQYJKoZIhvcNAQEFBQAD
// SIG // ggEBADK6pQkwkK45U28LOLA7FnXxWBqfHSg25lPn9o0z
// SIG // EIh1RB1hYZdFnuCJ3hU2UjSRZVEcGDZHnh4YwjPIMC7g
// SIG // 2OFRt3dgj2NL55Mu7N5Ic357FI1ghQbjtYMW7Y5nUi/B
// SIG // W3CdXTXZOfXZ1h66r59kSwnjBlqL9BwYeFVKDOLCXY44
// SIG // +754utMK/LySbZ81FYBN8CSkyHlEqFY+BHNfpUvfqqfy
// SIG // +DuqJvPM4+Pq+aMb2DpHweOeJKD4izbnpJwfEexkQ1rh
// SIG // 1ufqQP9ZUZRHFMAF9QRgZNj2EEQh82M6rL50H4bhHyTw
// SIG // CkMxKpPgn8wI6iLZ1c7UQmwizrCGeGWh0lLcEhMxggL1
// SIG // MIIC8QIBATCBkzB8MQswCQYDVQQGEwJVUzETMBEGA1UE
// SIG // CBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEe
// SIG // MBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSYw
// SIG // JAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0Eg
// SIG // MjAxMAITMwAAALCG6ZIgCl3q+AAAAAAAsDANBglghkgB
// SIG // ZQMEAgEFAKCCATIwGgYJKoZIhvcNAQkDMQ0GCyqGSIb3
// SIG // DQEJEAEEMC8GCSqGSIb3DQEJBDEiBCDc6He7qtjsXUzl
// SIG // NP1wIC6GEa0dS6UQoft8TOB6XwiXrDCB4gYLKoZIhvcN
// SIG // AQkQAgwxgdIwgc8wgcwwgbEEFLyE+51bEtrHNoU7iGae
// SIG // oxYY1cwcMIGYMIGApH4wfDELMAkGA1UEBhMCVVMxEzAR
// SIG // BgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1v
// SIG // bmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlv
// SIG // bjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAg
// SIG // UENBIDIwMTACEzMAAACwhumSIApd6vgAAAAAALAwFgQU
// SIG // iuxRJ82n69xsA0yzhw1JuPv2LOgwDQYJKoZIhvcNAQEL
// SIG // BQAEggEADJzvyBMutQmJtDAXZsd7Qsf4S/myApVEKeuZ
// SIG // es0DkVlD6NQP+y8B5bv1lBgV/pv7xIPl61DsoVjxIUgu
// SIG // eWxASzWIp28W6lRO+cKyOr9UO6AqqF5cSMKYpbeMLKTU
// SIG // /QVIMTqQyAsWW1VoHfvkxVTXUGld64CTXs7O0phbWiZv
// SIG // puh1dN+yDMUJdesuPNqA8ZNpnCS1ynBHoALldUQCmNip
// SIG // r6Sy+EZvi1YGxgGwMy/nX4A22aoIbEqKRrM/PZGDvdLC
// SIG // t9LrcMHczfR/EF/dA9CS4EjnUWEwcknaBoC3LOTk6MAS
// SIG // vaciNtKDIyhe6XskvfYvXgeZ8ftToyAPfxm7xC1iMA==
// SIG // End signature block
